using System;
using System.Diagnostics;

namespace Implab {
    class PromiseFuncReaction<TRet> : IResolvable {
        readonly Deferred<TRet> m_next;

        readonly IDispatcher m_dispatcher;

        readonly Action<Deferred<TRet>> m_fulfilled;

        readonly Action<Exception, Deferred<TRet>> m_rejected;

        public IPromise<TRet> Promise {
            get { return m_next.Promise; }
        }

        public PromiseFuncReaction(Action<Deferred<TRet>> fulfilled, Action<Exception, Deferred<TRet>> rejected, Deferred<TRet> next, IDispatcher dispatcher) {
            m_next = next;
            m_fulfilled = fulfilled;
            m_rejected = rejected;
            m_dispatcher = dispatcher;
        }

        public void Resolve() {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(ResolveImpl);
                else
                    ResolveImpl();
            } else {
                m_next.Resolve(default(TRet));
            }
        }

        void ResolveImpl () {
            m_fulfilled(m_next);
        }

        public void Reject(Exception error) {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(RejectImpl, error);
                else
                    RejectImpl(error);
            } else {
                m_next.Reject(error);
            }
        }

        void RejectImpl(Exception error) {
            m_rejected(error, m_next);
        }

        public static PromiseFuncReaction<TRet> Create(Func<TRet> fulfilled, Func<Exception, TRet> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TRet> Create(Func<IPromise<TRet>> fulfilled, Func<Exception, TRet> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TRet> Create(Func<TRet> fulfilled, Func<Exception, IPromise<TRet>> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TRet> Create(Func<IPromise<TRet>> fulfilled, Func<Exception, IPromise<TRet>> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }
    }
}